/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.plugin.core.analysis;

import docking.options.editor.FileChooserEditor;
import docking.options.editor.StringWithChoicesEditor;
import generic.jar.ResourceFile;
import ghidra.app.cmd.function.ApplyFunctionDataTypesCmd;
import ghidra.app.plugin.core.analysis.AutoAnalysisManager;
import ghidra.app.plugin.core.analysis.ProjectPathChooserEditor;
import ghidra.app.plugin.core.datamgr.util.DataTypeArchiveUtility;
import ghidra.app.services.AbstractAnalyzer;
import ghidra.app.services.AnalysisPriority;
import ghidra.app.services.AnalyzerType;
import ghidra.app.services.DataTypeManagerService;
import ghidra.app.util.importer.MessageLog;
import ghidra.framework.Application;
import ghidra.framework.model.DomainFile;
import ghidra.framework.model.DomainFileFilter;
import ghidra.framework.model.DomainObject;
import ghidra.framework.model.ProjectData;
import ghidra.framework.options.OptionType;
import ghidra.framework.options.Options;
import ghidra.program.model.address.AddressSetView;
import ghidra.program.model.data.DataTypeManager;
import ghidra.program.model.data.FileDataTypeManager;
import ghidra.program.model.listing.DataTypeArchive;
import ghidra.program.model.listing.Program;
import ghidra.program.model.symbol.SourceType;
import ghidra.util.Msg;
import ghidra.util.exception.VersionException;
import ghidra.util.task.TaskMonitor;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

public class ApplyDataArchiveAnalyzer
extends AbstractAnalyzer {
    private static final String NAME = "Apply Data Archives";
    private static final String DESCRIPTION = "Apply known data type archives based on program information.";
    private static final String OPTION_NAME_CREATE_BOOKMARKS = "Create Analysis Bookmarks";
    private static final String OPTION_DESCRIPTION_CREATE_BOOKMARKS = "If checked, an analysis bookmark will be created at each symbol address where multiple function definitions were found and not applied.";
    private static final boolean OPTION_DEFAULT_CREATE_BOOKMARKS_ENABLED = true;
    private static final String OPTION_NAME_ARCHIVE_CHOOSER = "Archive Chooser";
    private static final String OPTION_DESCRIPTION_ARCHIVE_CHOOSER = "Specifies the data type archive to apply";
    private static final String OPTION_NAME_GDT_FILEPATH = "GDT User File Archive Path";
    private static final String OPTION_DESCRIPTION_GDT_FILEPATH = "Path to a user-supplied data type archive .gdt file, only valid when 'Archive Chooser' is '[User-File-Archive]'";
    private static final String OPTION_NAME_PROJECT_PATH = "User Project Archive Path";
    private static final String OPTION_DESCRIPTION_PROJECT_PATH = "Path to a user-supplied data type archive located in the project, only valid when 'Archive Chooser' is '[User-Project-Archive]'";
    private static final String CHOOSER_AUTO_DETECT = "[Auto-Detect]";
    private static final String CHOOSER_USER_FILE_ARCHIVE = "[User-File-Archive]";
    private static final String CHOOSER_USER_PROJECT_ARCHIVE = "[User-Project-Archive]";
    private boolean createBookmarksEnabled = true;
    private String archiveChooser = "[Auto-Detect]";
    private File userGdtFileArchive;
    private String userProjectArchive;
    private Map<String, ResourceFile> builtinGDTs = ApplyDataArchiveAnalyzer.getBuiltInGdts();
    private DataTypeManagerService dtmService;
    private static final DomainFileFilter DATATYPEARCHIVE_PROJECT_FILTER = df -> DataTypeArchive.class.isAssignableFrom(df.getDomainObjectClass());

    public ApplyDataArchiveAnalyzer() {
        super(NAME, DESCRIPTION, AnalyzerType.BYTE_ANALYZER);
        this.setPriority(AnalysisPriority.FUNCTION_ID_ANALYSIS.after());
        this.setDefaultEnablement(true);
    }

    @Override
    public boolean getDefaultEnablement(Program program) {
        if ("golang".equals(program.getCompilerSpec().getCompilerSpecID().toString())) {
            return false;
        }
        return super.getDefaultEnablement(program);
    }

    @Override
    public boolean added(Program program, AddressSetView set, TaskMonitor monitor, MessageLog log) {
        this.dtmService = AutoAnalysisManager.getAnalysisManager(program).getDataTypeManagerService();
        List<DataTypeManager> managerList = this.getDataTypeArchives(program, log, monitor);
        if (!managerList.isEmpty()) {
            monitor.setMessage("Applying Function Signatures...");
            ApplyFunctionDataTypesCmd cmd = new ApplyFunctionDataTypesCmd(managerList, set, SourceType.IMPORTED, false, this.createBookmarksEnabled);
            cmd.applyTo((DomainObject)program, monitor);
            for (DataTypeManager dtm : managerList) {
                Msg.info((Object)this, (Object)"Applied data type archive: %s".formatted(dtm.getName()));
            }
        }
        return true;
    }

    @Override
    public void registerOptions(Options options, Program program) {
        options.registerOption(OPTION_NAME_CREATE_BOOKMARKS, (Object)this.createBookmarksEnabled, null, OPTION_DESCRIPTION_CREATE_BOOKMARKS);
        ArrayList<String> chooserList = new ArrayList<String>();
        chooserList.add(CHOOSER_AUTO_DETECT);
        chooserList.add(CHOOSER_USER_FILE_ARCHIVE);
        chooserList.add(CHOOSER_USER_PROJECT_ARCHIVE);
        chooserList.addAll(this.builtinGDTs.keySet());
        options.registerOption(OPTION_NAME_ARCHIVE_CHOOSER, OptionType.STRING_TYPE, (Object)CHOOSER_AUTO_DETECT, null, OPTION_DESCRIPTION_ARCHIVE_CHOOSER, () -> new StringWithChoicesEditor(chooserList));
        options.registerOption(OPTION_NAME_GDT_FILEPATH, OptionType.FILE_TYPE, null, null, OPTION_DESCRIPTION_GDT_FILEPATH, () -> new FileChooserEditor(FileDataTypeManager.GDT_FILEFILTER));
        options.registerOption(OPTION_NAME_PROJECT_PATH, OptionType.STRING_TYPE, null, null, OPTION_DESCRIPTION_PROJECT_PATH, () -> new ProjectPathChooserEditor("Choose Data Type Archive", DATATYPEARCHIVE_PROJECT_FILTER));
    }

    @Override
    public void optionsChanged(Options options, Program program) {
        this.createBookmarksEnabled = options.getBoolean(OPTION_NAME_CREATE_BOOKMARKS, this.createBookmarksEnabled);
        this.archiveChooser = options.getString(OPTION_NAME_ARCHIVE_CHOOSER, this.archiveChooser);
        this.userGdtFileArchive = options.getFile(OPTION_NAME_GDT_FILEPATH, this.userGdtFileArchive);
        this.userProjectArchive = options.getString(OPTION_NAME_PROJECT_PATH, this.userProjectArchive);
    }

    private List<DataTypeManager> getDataTypeArchives(Program program, MessageLog log, TaskMonitor monitor) {
        switch (this.archiveChooser) {
            case "[Auto-Detect]": {
                return this.getAutoDTMs(program, log, monitor);
            }
            case "[User-File-Archive]": {
                return this.openUserFileArchive(this.userGdtFileArchive, log);
            }
            case "[User-Project-Archive]": {
                return this.openUserProjectArchive(this.userProjectArchive, program, log, monitor);
            }
        }
        return this.openBuiltinGDT(this.archiveChooser, log);
    }

    private List<DataTypeManager> getAutoDTMs(Program program, MessageLog log, TaskMonitor monitor) {
        List<String> archiveList = DataTypeArchiveUtility.getArchiveList(program);
        ArrayList<DataTypeManager> result = new ArrayList<DataTypeManager>();
        monitor.initialize((long)archiveList.size());
        for (String archiveName : archiveList) {
            if (monitor.isCancelled()) break;
            try {
                DataTypeManager dtm = this.dtmService.openDataTypeArchive(archiveName);
                if (dtm == null) {
                    log.appendMsg(NAME, "Failed to locate data type archive: " + archiveName);
                    continue;
                }
                result.add(dtm);
            }
            catch (Exception e) {
                Throwable cause = e.getCause();
                if (cause instanceof VersionException) {
                    log.appendMsg(NAME, "Unable to open archive %s: %s".formatted(archiveName, cause.toString()));
                    continue;
                }
                String msg = Objects.requireNonNullElse(e.getMessage(), e.toString());
                log.appendMsg(NAME, "Unexpected Error opening archive %s: %s".formatted(archiveName, msg));
            }
        }
        return result;
    }

    private List<DataTypeManager> openBuiltinGDT(String gdtName, MessageLog log) {
        ResourceFile gdtFile = this.builtinGDTs.get(gdtName);
        if (gdtFile == null) {
            log.appendMsg("Unknown built-in archive: %s".formatted(gdtName));
            return List.of();
        }
        try {
            return List.of(this.dtmService.openArchive(gdtFile, false));
        }
        catch (Exception e) {
            Throwable cause = e.getCause();
            if (cause instanceof VersionException) {
                log.appendMsg(NAME, "Unable to open archive %s: %s".formatted(gdtName, cause.toString()));
            } else {
                String msg = Objects.requireNonNullElse(e.getMessage(), e.toString());
                log.appendMsg(NAME, "Unexpected Error opening archive %s: %s".formatted(gdtName, msg));
            }
            return List.of();
        }
    }

    private List<DataTypeManager> openUserFileArchive(File gdtFile, MessageLog log) {
        if (gdtFile == null) {
            return List.of();
        }
        if (!gdtFile.isFile()) {
            log.appendMsg("Missing archive: %s".formatted(gdtFile));
            return List.of();
        }
        try {
            return List.of(this.dtmService.openArchive(new ResourceFile(gdtFile), false));
        }
        catch (Exception e) {
            Throwable cause = e.getCause();
            if (cause instanceof VersionException) {
                log.appendMsg(NAME, "Unable to open archive %s: %s".formatted(gdtFile, cause.toString()));
            } else {
                String msg = Objects.requireNonNullElse(e.getMessage(), e.toString());
                log.appendMsg(NAME, "Unexpected Error opening archive %s: %s".formatted(gdtFile, msg));
            }
            return List.of();
        }
    }

    private List<DataTypeManager> openUserProjectArchive(String filename, Program program, MessageLog log, TaskMonitor monitor) {
        if (filename == null || filename.isBlank()) {
            return List.of();
        }
        ProjectData projectData = program.getDomainFile().getParent().getProjectData();
        DomainFile gdtDomainFile = projectData.getFile(filename);
        if (gdtDomainFile == null) {
            log.appendMsg("Missing project archive: %s".formatted(filename));
            return List.of();
        }
        if (!DataTypeArchive.class.isAssignableFrom(gdtDomainFile.getDomainObjectClass())) {
            log.appendMsg("Bad project file type: %s".formatted(filename));
            return List.of();
        }
        try {
            return List.of(this.dtmService.openArchive(gdtDomainFile, monitor));
        }
        catch (Exception e) {
            Throwable cause = e.getCause();
            if (cause instanceof VersionException) {
                log.appendMsg(NAME, "Unable to open project archive %s: %s".formatted(filename, cause.toString()));
            } else {
                String msg = Objects.requireNonNullElse(e.getMessage(), e.toString());
                log.appendMsg(NAME, "Unexpected Error opening project archive %s: %s".formatted(filename, msg));
            }
            return List.of();
        }
    }

    private static Map<String, ResourceFile> getBuiltInGdts() {
        return Application.findFilesByExtensionInApplication((String)".gdt").stream().collect(Collectors.toMap(f -> f.getName(), f -> f));
    }
}

