/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.knn.indices;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalCause;
import com.google.common.cache.RemovalNotification;
import java.time.Instant;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.cluster.service.ClusterService;
import org.opensearch.core.common.unit.ByteSizeValue;
import org.opensearch.knn.common.KNNConstants;
import org.opensearch.knn.index.KNNSettings;
import org.opensearch.knn.indices.Model;
import org.opensearch.knn.indices.ModelDao;

public final class ModelCache {
    private static Logger logger = LogManager.getLogger(ModelCache.class);
    private static ModelCache instance;
    private static ModelDao modelDao;
    private static ClusterService clusterService;
    private Cache<String, Model> cache;
    private long cacheSizeInKB = ((ByteSizeValue)KNNSettings.MODEL_CACHE_SIZE_LIMIT_SETTING.get(clusterService.getSettings())).getKb();
    private Instant evictedDueToSizeAt;

    public static synchronized ModelCache getInstance() {
        if (instance == null) {
            instance = new ModelCache();
        }
        return instance;
    }

    public static void initialize(ModelDao modelDao, ClusterService clusterService) {
        ModelCache.modelDao = modelDao;
        ModelCache.clusterService = clusterService;
    }

    public synchronized void rebuild() {
        this.cache.invalidateAll();
        this.initCache();
    }

    protected ModelCache() {
        clusterService.getClusterSettings().addSettingsUpdateConsumer(KNNSettings.MODEL_CACHE_SIZE_LIMIT_SETTING, it -> {
            this.cacheSizeInKB = it.getKb();
            this.rebuild();
        });
        this.initCache();
    }

    private void initCache() {
        CacheBuilder cacheBuilder = CacheBuilder.newBuilder().recordStats().concurrencyLevel(1).removalListener(this::onRemoval).maximumWeight(this.cacheSizeInKB).expireAfterAccess((long)KNNConstants.MODEL_CACHE_EXPIRE_AFTER_ACCESS_TIME_MINUTES.intValue(), TimeUnit.MINUTES).weigher((k, v) -> Math.toIntExact(this.getModelLengthInKB((Model)v)));
        this.cache = cacheBuilder.build();
    }

    private void onRemoval(RemovalNotification<String, Model> removalNotification) {
        if (RemovalCause.SIZE == removalNotification.getCause()) {
            this.updateEvictedDueToSizeAt();
        }
        logger.info("[KNN] Model Cache evicted. Key {}, Reason: {}", removalNotification.getKey(), (Object)removalNotification.getCause());
    }

    public Instant getEvictedDueToSizeAt() {
        return this.evictedDueToSizeAt;
    }

    private void updateEvictedDueToSizeAt() {
        this.evictedDueToSizeAt = Instant.now();
    }

    public Model get(String modelId) {
        try {
            return (Model)this.cache.get((Object)modelId, () -> modelDao.get(modelId));
        }
        catch (ExecutionException ee) {
            throw new IllegalStateException("Unable to retrieve model binary for \"" + modelId + "\": " + ee);
        }
    }

    public long getTotalWeightInKB() {
        return this.cache.asMap().values().stream().map(this::getModelLengthInKB).reduce(0L, Long::sum);
    }

    public void remove(String modelId) {
        this.cache.invalidate((Object)modelId);
    }

    public boolean contains(String modelId) {
        return this.cache.asMap().containsKey(modelId);
    }

    public void removeAll() {
        this.cache.invalidateAll();
    }

    private Long getModelLengthInKB(Model model) {
        return (long)(model.getLength() / KNNConstants.BYTES_PER_KILOBYTES) + 1L;
    }
}

