"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = _default;

var _configSchema = require("@osd/config-schema");

var _common = require("../../common");

var _helpers = require("./utils/helpers");

var _createReportDefinition = require("./lib/createReportDefinition");

var _backendToUi = require("./utils/converters/backendToUi");

var _updateReportDefinition = require("./lib/updateReportDefinition");

var _constants = require("./utils/constants");

var _metricHelper = require("./utils/metricHelper");

var _validationHelper = require("../../server/utils/validationHelper");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
function _default(router, config) {
  const protocol = config.get('osd_server', 'protocol');
  const hostname = config.get('osd_server', 'hostname');
  const port = config.get('osd_server', 'port');
  const basePath = config.osdConfig.get('server', 'basePath'); // Create report Definition

  router.post({
    path: `${_common.API_PREFIX}/reportDefinition`,
    validate: {
      body: _configSchema.schema.any()
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report_definition', 'create', 'count');
    let reportDefinition = request.body; //@ts-ignore

    const logger = context.reporting_plugin.logger; // input validation

    try {
      reportDefinition.report_params.core_params.origin = `${protocol}://${hostname}:${port}${basePath}`;
      reportDefinition = await (0, _validationHelper.validateReportDefinition)(context.core.opensearch.legacy.client, reportDefinition, basePath);
    } catch (error) {
      logger.error(`Failed input validation for create report definition ${error}`);
      (0, _metricHelper.addToMetric)('report_definition', 'create', 'user_error');
      return response.badRequest({
        body: error
      });
    } // save metadata


    try {
      const res = await (0, _createReportDefinition.createReportDefinition)(request, context, reportDefinition);
      return response.ok({
        body: {
          state: 'Report definition created',
          scheduler_response: res
        }
      });
    } catch (error) {
      logger.error(`Failed to create report definition: ${error}`);
      (0, _metricHelper.addToMetric)('report_definition', 'create', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  }); // Update report definition by id

  router.put({
    path: `${_common.API_PREFIX}/reportDefinitions/{reportDefinitionId}`,
    validate: {
      body: _configSchema.schema.any(),
      params: _configSchema.schema.object({
        reportDefinitionId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report_definition', 'update', 'count');
    let reportDefinition = request.body; //@ts-ignore

    const logger = context.reporting_plugin.logger; // input validation

    try {
      reportDefinition.report_params.core_params.origin = request.headers.origin;
      reportDefinition = await (0, _validationHelper.validateReportDefinition)(context.core.opensearch.legacy.client, reportDefinition, basePath);
    } catch (error) {
      logger.error(`Failed input validation for update report definition ${error}`);
      (0, _metricHelper.addToMetric)('report_definition', 'update', 'user_error');
      return response.badRequest({
        body: error
      });
    } // Update report definition metadata


    try {
      const opensearchResp = await (0, _updateReportDefinition.updateReportDefinition)(request, context, reportDefinition);
      return response.ok({
        body: {
          state: 'Report definition updated',
          scheduler_response: opensearchResp
        }
      });
    } catch (error) {
      logger.error(`Failed to update report definition: ${error}`);
      (0, _metricHelper.addToMetric)('report_definition', 'update', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  }); // get all report definitions details

  router.get({
    path: `${_common.API_PREFIX}/reportDefinitions`,
    validate: {
      query: _configSchema.schema.object({
        fromIndex: _configSchema.schema.maybe(_configSchema.schema.number()),
        maxItems: _configSchema.schema.maybe(_configSchema.schema.number())
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report_definition', 'list', 'count');
    const {
      fromIndex,
      maxItems
    } = request.query;

    try {
      // @ts-ignore
      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request);
      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.getReportDefinitions', {
        fromIndex: fromIndex,
        maxItems: maxItems || _constants.DEFAULT_MAX_SIZE
      });
      const reportDefinitionsList = (0, _backendToUi.backendToUiReportDefinitionsList)(opensearchResp.reportDefinitionDetailsList, basePath);
      return response.ok({
        body: {
          data: reportDefinitionsList
        }
      });
    } catch (error) {
      //@ts-ignore
      context.reporting_plugin.logger.error(`Failed to get report definition details: ${error}`);
      (0, _metricHelper.addToMetric)('report_definition', 'list', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  }); // get report definition detail by id

  router.get({
    path: `${_common.API_PREFIX}/reportDefinitions/{reportDefinitionId}`,
    validate: {
      params: _configSchema.schema.object({
        reportDefinitionId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report_definition', 'info', 'count');

    try {
      // @ts-ignore
      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request);
      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.getReportDefinitionById', {
        reportDefinitionId: request.params.reportDefinitionId
      });
      const reportDefinition = (0, _backendToUi.backendToUiReportDefinition)(opensearchResp.reportDefinitionDetails, basePath);
      return response.ok({
        body: {
          report_definition: reportDefinition
        }
      });
    } catch (error) {
      //@ts-ignore
      context.reporting_plugin.logger.error(`Failed to get single report details: ${error}`);
      (0, _metricHelper.addToMetric)('report_definition', 'info', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  }); // Delete report definition by id

  router.delete({
    path: `${_common.API_PREFIX}/reportDefinitions/{reportDefinitionId}`,
    validate: {
      params: _configSchema.schema.object({
        reportDefinitionId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report_definition', 'delete', 'count');

    try {
      // @ts-ignore
      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request);
      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.deleteReportDefinitionById', {
        reportDefinitionId: request.params.reportDefinitionId
      });
      return response.ok({
        body: {
          state: 'Report definition deleted',
          opensearch_response: opensearchResp
        }
      });
    } catch (error) {
      //@ts-ignore
      context.reporting_plugin.logger.error(`Failed to delete report definition: ${error}`);
      (0, _metricHelper.addToMetric)('report_definition', 'delete', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  });
}

module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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