import Bluebird from "bluebird";
import { Defer } from "../promiseutil";
export interface QueueItem<T> {
    id: string;
    item: T;
    defer: Defer<unknown>;
}
export type QueueProcessFn<T> = (item: T) => Promise<unknown> | void;
export declare class Queue<T> {
    private processFn;
    private intervalMs?;
    private queue;
    private processing;
    private onceFreeDefers;
    private consume;
    /**
     * Construct a new Queue which will process items FIFO.
     * @param {Function} processFn The function to invoke when the item being processed
     * is in its critical section. Only 1 item at any one time will be calling this function.
     * The function should return a Promise which is resolved/rejected when the next item
     * can be taken from the queue.
     * @param {integer} intervalMs Optional. If provided and > 0, this queue will be serviced
     * at an interval of intervalMs. Otherwise, items will be processed as soon as they become
     * the first item in the queue to be processed.
     */
    constructor(processFn: QueueProcessFn<T>, intervalMs?: number | undefined);
    /**
     * Return the length of the queue, including the currently processed item.
     * @return {Number} The length of the queue.
     */
    size(): number;
    /**
     * Return a promise which is resolved when this queue is free (0 items in queue).
     * @return {Promise<Queue>} Resolves to the Queue itself.
     */
    onceFree(): Promise<unknown>;
    private fireOnceFree;
    /**
     * Queue up a request for the critical section function.
     * @param {string} id An ID to associate with this request. If there is already a
     * request with this ID, the promise for that request will be returned.
     * @param {*} thing The item to enqueue. It will be passed verbatim to the critical
     * section function passed in the constructor.
     * @return {Promise} A promise which will be resolved/rejected when the queued item
     * has been processed.
     */
    enqueue(id: string, thing: T): Bluebird<unknown>;
    private retry;
    private coConsume;
    killAll(): void;
}
